let lastClickedElement = null;

// Track right-clicked element
document.addEventListener("contextmenu", (event) => {
    lastClickedElement = event.target;
}, true);

// Listen for messages from background
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.type === "AXIOM_ACTION") {
        handleAction(request);
    }
});

async function handleAction(data) {
    const storageData = await chrome.storage.sync.get({
        immediateReplace: false,
        selectionOnly: false
    });

    const settings = {
        ...storageData,
        apiUrl: "https://api.axiomai.dev/rewrite",
        clientToken: "axiom_98237492837492387445985645678156062419656179565441745688"
    };

    // determine text
    let text = "";
    let mode = "selection"; // or 'element'
    let range = null;

    const selection = window.getSelection();
    if (selection.toString().trim().length > 0) {
        text = selection.toString();
        if (selection.rangeCount > 0) {
            range = selection.getRangeAt(0);
        }
    } else if (!settings.selectionOnly && lastClickedElement) {
        // Fallback to element text
        // Try to get a block element
        let el = lastClickedElement;
        while (el && el !== document.body && !isBlock(el)) {
            el = el.parentElement;
        }
        if (el) {
            text = el.innerText;
            lastClickedElement = el; // update to the block
            mode = "element";
        }
    }

    // Final fallback to what background saw (if any)
    if (!text && data.selectionText) {
        text = data.selectionText;
    }

    if (!text || text.trim().length === 0) {
        alert("AxiomAI: No text selected or found.");
        return;
    }

    // Token check removed (hardcoded)

    // Function to perform replacement
    const applyReplacement = (newText) => {
        if (mode === "selection" && range) {
            range.deleteContents();
            range.insertNode(document.createTextNode(newText));
            window.getSelection().removeAllRanges();
        } else if (mode === "element" && lastClickedElement) {
            lastClickedElement.innerText = newText;
        } else {
            // If we lost context, copy to clipboard
            navigator.clipboard.writeText(newText);
            alert("Context lost. Text copied to clipboard instead.");
        }
    };

    // UI Manager
    const overlay = new AxiomOverlay(text);
    overlay.show();

    try {
        const response = await fetch(settings.apiUrl, {
            method: "POST",
            headers: {
                "Content-Type": "application/json",
                "X-Client-Token": settings.clientToken
            },
            body: JSON.stringify({
                action: data.action,
                text: text,
                url: data.pageUrl,
                title: data.pageTitle
            })
        });

        const result = await response.json();

        if (!response.ok) {
            let msg = result.error || "Unknown error occurred.";
            if (response.status === 429) {
                msg = "Hourly rate limit reached. Please wait a bit before trying again.";
            }
            overlay.showError(msg);
            return;
        }

        // Extract headers
        // Extract headers
        const limitStr = response.headers.get("RateLimit-Limit");
        const remainingStr = response.headers.get("RateLimit-Remaining");
        const resetStr = response.headers.get("RateLimit-Reset");

        let quota = null;
        if (limitStr && remainingStr) {
            quota = {
                total: parseInt(limitStr),
                remaining: parseInt(remainingStr),
                reset: resetStr ? parseInt(resetStr) : null
            };
        }

        if (settings.immediateReplace) {
            applyReplacement(result.outputText);
            overlay.close();
        } else {
            // Pass action to handle UI (toggle, hide replace)
            overlay.showResult(result.outputText, data.action, (finalText) => {
                applyReplacement(finalText);
                overlay.close();
            }, quota);
        }

    } catch (err) {
        console.error(err);
        overlay.showError("Network error. Check your connection or the server status.");
    }
}

// Helper to check for block elements
function isBlock(el) {
    const style = window.getComputedStyle(el);
    return ['block', 'flex', 'grid', 'table-cell'].includes(style.display) && el.innerText.trim().length > 0;
}

// --- Overlay Class ---
// --- Diff Algorithm (Simple Word-Level LCS) ---
function computeDiff(oldText, newText) {
    const oldWords = oldText.split(/\s+/);
    const newWords = newText.split(/\s+/);

    // 1. Compute LCS Matrix
    const dp = Array(oldWords.length + 1).fill(0).map(() => Array(newWords.length + 1).fill(0));

    for (let i = 1; i <= oldWords.length; i++) {
        for (let j = 1; j <= newWords.length; j++) {
            if (oldWords[i - 1] === newWords[j - 1]) {
                dp[i][j] = dp[i - 1][j - 1] + 1;
            } else {
                dp[i][j] = Math.max(dp[i - 1][j], dp[i][j - 1]);
            }
        }
    }

    // 2. Backtrack to find diff
    let i = oldWords.length;
    let j = newWords.length;
    const result = [];

    while (i > 0 || j > 0) {
        if (i > 0 && j > 0 && oldWords[i - 1] === newWords[j - 1]) {
            result.unshift(escapeHtml(oldWords[i - 1]));
            i--;
            j--;
        } else if (j > 0 && (i === 0 || dp[i][j - 1] >= dp[i - 1][j])) {
            result.unshift(`<span class="axiom-diff-ins">${escapeHtml(newWords[j - 1])}</span>`);
            j--;
        } else {
            result.unshift(`<span class="axiom-diff-del">${escapeHtml(oldWords[i - 1])}</span>`);
            i--;
        }
    }

    return result.join(" ");
}

// --- Overlay Class ---
class AxiomOverlay {
    constructor(originalText) {
        this.originalText = originalText;
        this.host = document.createElement('div');
        this.host.id = 'axiom-overlay-host';
        this.host.innerHTML = `
      <div class="axiom-backdrop"></div>
      <div class="axiom-modal">
        <div class="axiom-header">
          <div class="axiom-title">AxiomAI Rewriter</div>
          <div class="axiom-close">&times;</div>
        </div>
        <div class="axiom-body">
          <div>
            <div class="axiom-section-label">Original</div>
            <div class="axiom-text-box">${escapeHtml(originalText)}</div>
            <div class="axiom-help-text">This is the text you selected or clicked on. It will be sent to the AI for processing.</div>
          </div>
          <div id="axiom-result-area">
            <div class="axiom-loading-container">
              <div class="axiom-spinner"></div>
              <div>Thinking...</div>
            </div>
          </div>
        </div>
        <div class="axiom-footer" id="axiom-footer">
          <button class="axiom-btn axiom-btn-secondary" id="axiom-cancel-btn">Cancel</button>
        </div>
      </div>
    `;
    }

    show() {
        document.body.appendChild(this.host);
        this.host.querySelector('.axiom-backdrop').addEventListener('click', () => this.close());
        this.host.querySelector('.axiom-close').addEventListener('click', () => this.close());
        this.host.querySelector('#axiom-cancel-btn').addEventListener('click', () => this.close());
    }

    showError(message) {
        const resultArea = this.host.querySelector('#axiom-result-area');
        resultArea.innerHTML = `
            <div class="axiom-section-label" style="color: #ef4444;">Error</div>
            <div class="axiom-text-box" style="border-color: #ef4444; background: #fef2f2; color: #b91c1c;">
                ${escapeHtml(message)}
            </div>
            <div class="axiom-help-text">Please try again later or contact support if this persists.</div>
        `;
        // Hide/Change footer buttons if needed, but Cancel is fine.
    }

    showResult(newText, action, onApply, quota = null) {
        const resultArea = this.host.querySelector('#axiom-result-area');
        const isAnalysis = action.startsWith('analyze_') || action.startsWith('seo_') && action !== 'seo_meta';

        // Header for Result Section (with Toggle if applicable)
        let headerHtml = `<div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:5px;">
                            <div class="axiom-section-label" style="margin-bottom:0;">Result</div>`;

        if (!isAnalysis) {
            headerHtml += `<label class="axiom-toggle-container">
                             <input type="checkbox" class="axiom-toggle-input" id="axiom-diff-toggle">
                             <span class="axiom-toggle-slider"></span>
                             Show Changes
                           </label>`;
        }
        headerHtml += `</div>`;

        // Rate Limit Info
        let quotaHtml = "";
        if (quota) {
            const { remaining, total, reset } = quota;
            let color = "#16a34a"; // green
            if (remaining < 10) color = "#dc2626"; // red
            else if (remaining < 30) color = "#ca8a04"; // yellow

            let resetText = "";
            if (reset) {
                // RateLimit-Reset is delta seconds in standardHeaders draft-7
                const resetDate = new Date(Date.now() + (reset * 1000));
                const timeString = resetDate.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                resetText = ` (Renew: ${timeString})`;
            }

            quotaHtml = `<div style="font-size:11px; color:#64748b; margin-top:4px; text-align:right;">
                            Quota: <span style="font-weight:bold; color:${color}">${remaining}/${total}</span>${resetText}
                         </div>`;
        }

        resultArea.innerHTML = `
           ${headerHtml}
           <div class="axiom-text-box" id="axiom-result-text" style="border-color: #007bff; background: #f0f7ff;">${escapeHtml(newText)}</div>
           ${quotaHtml}
           <div class="axiom-help-text">
             ${isAnalysis ? "This is the analysis result. usage suggestions: Copy key insights." : "This is the AI-rewritten text. <strong>Note: Changes are local and will be lost if you refresh the page.</strong>"}
           </div>
        `;

        // Handle Diff Toggle Logic
        if (!isAnalysis) {
            const toggle = this.host.querySelector('#axiom-diff-toggle');
            const textBox = this.host.querySelector('#axiom-result-text');
            const plainHtml = escapeHtml(newText);
            const diffHtml = computeDiff(this.originalText, newText);

            toggle.addEventListener('change', (e) => {
                textBox.innerHTML = e.target.checked ? diffHtml : plainHtml;
            });
        }

        const footer = this.host.querySelector('#axiom-footer');

        // Copy Button
        const copyBtn = document.createElement('button');
        copyBtn.className = 'axiom-btn axiom-btn-secondary';
        copyBtn.innerText = 'Copy';
        copyBtn.onclick = () => {
            navigator.clipboard.writeText(newText);
            copyBtn.innerText = 'Copied!';
            setTimeout(() => copyBtn.innerText = 'Copy', 1500);
        };

        footer.insertBefore(copyBtn, footer.firstChild);

        // Replace Button (Only if NOT analysis)
        if (!isAnalysis) {
            const applyBtn = document.createElement('button');
            applyBtn.className = 'axiom-btn axiom-btn-primary';
            applyBtn.innerText = 'Replace';
            applyBtn.onclick = () => {
                onApply();
                this.close();
            };
            footer.appendChild(applyBtn);
        }
    }



    close() {
        if (this.host.parentElement) {
            this.host.parentElement.removeChild(this.host);
        }
    }
}

function escapeHtml(text) {
    if (!text) return "";
    return text
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
}
